<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Password;


class AuthBasicController extends Controller
{
    public function register(Request $request)
    {
        $data = $request->validate([
            'username'   => ['required','alpha_dash','min:3','max:30','unique:users,username'],
            'first_name' => ['required','string','max:100'],
            'last_name'  => ['nullable','string','max:100'],
            'email'      => ['required','email','max:190','unique:users,email'],
            'whatsapp'   => ['nullable','regex:/^\+?[0-9]{8,15}$/'], // + dan 8-15 digit
            'password'   => ['required','string','min:6','confirmed','regex:/^(?=.*[A-Z])(?=.*\d)[A-Za-z\d]{6,}$/'], // butuh password_confirmation
        ], [
            'whatsapp.regex' => 'Format nomor WhatsApp tidak valid.',
        ]);

        $user = User::create([
            'username'   => $data['username'],
            'first_name' => $data['first_name'],
            'last_name'  => $data['last_name'],
            // gabungkan name penuh (boleh dipakai di tampilan)
            'name'       => trim($data['first_name'].' '.($data['last_name'] ?? '')),
            'email'      => $data['email'],
            'whatsapp'   => $data['whatsapp'] ?? null,
            'password'   => Hash::make($data['password']),
        ]);

        $token = JWTAuth::fromUser($user);

        return response()->json([
            'message'      => 'Register berhasil',
            'token' => $token,
            'token_type'   => 'Bearer',
            'expires_in'   => auth('api')->factory()->getTTL() * 60,
            'user'         => $user,
        ], 201);
    }

    public function login(Request $request)
    {
        $cred = $request->validate([
            'username'    => ['required','string','max:190'], // username atau name
            'password' => ['required','string'],
        ]);

        // 1) Cari berdasarkan username dulu (unik & direkomendasikan)
        $user = User::where('username', $cred['username'])->first();

        // 2) Jika tidak ketemu dan kamu memang mau izinkan "name", coba exact match name.
        if (!$user) {
            $user = User::where('name', $cred['username'])->first();
        }

        if (!$user || !Hash::check($cred['password'], $user->password ?? '')) {
            return response()->json(['error' => 'Password yang kamu masukan salah'], 401);
        }

        $token = auth('api')->fromUser($user);

        return response()->json([
            'message'      => 'Login berhasil',
            'token' => $token,
            'token_type'   => 'Bearer',
            'expires_in'   => auth('api')->factory()->getTTL() * 60,
            'user'         => $user,
        ]);
    }

    public function sendResetLink(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
        ]);

        $status = Password::sendResetLink($request->only('email'));

        if ($status === Password::RESET_LINK_SENT) {
            return response()->json(['message' => 'Link reset password telah dikirim ke email Anda.']);
        }

        return response()->json(['message' => 'Gagal mengirim link reset password.'], 500);
    }

    public function reset(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
            'token' => 'required',
            'password' => [
                'required',
                'string',
                'min:8',
                'confirmed',
                'regex:/^(?=.*[A-Z])(?=.*\d)[A-Za-z\d]{8,}$/'
            ],
        ], [
            'password.regex' => 'Password harus mengandung minimal 1 huruf besar dan 1 angka.',
        ]);

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->forceFill([
                    'password' => Hash::make($password),
                ])->save();
            }
        );

        if ($status === Password::PASSWORD_RESET) {
            return response()->json(['message' => 'Password berhasil direset. Silakan login kembali.']);
        }

        return response()->json(['message' => 'Token reset password tidak valid atau sudah kedaluwarsa.'], 400);
    }
}
